/*
 * Decompiled with CFR 0.152.
 */
package ch.varial.accounting.domain.payment.iso.resolver;

import ch.varial.accounting.domain.payment.iso.data.RemittanceInformation;
import ch.varial.accounting.domain.payment.iso.data.TransactionType;
import ch.varial.accounting.domain.payment.iso.format.Iso20022ChFieldFormatter;
import ch.varial.accounting.domain.payment.iso.util.Iso20022ChPaymentUtils;
import com.varial.accounting.domain.openitem.OpenItem;
import com.varial.accounting.domain.openitem.SwissReferenceType;
import com.varial.accounting.domain.payment.FinancePayment;
import com.varial.accounting.domain.payment.FinancePaymentSession;
import com.varial.accounting.domain.payment.GeneratePaymentRemittance;
import com.varial.accounting.domain.payment.PaymentHelper;
import com.varial.base.exception.BusinessDomainException;
import com.varial.framework.domain.paymentbase.Payment;
import com.varial.framework.domain.paymentbase.PaymentSession;
import com.varial.interfaces.IBankAccount;
import com.varial.serverbase.mapper.StringMapper;
import java.util.Optional;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;

public class RemittanceResolver {
    private static final int ESR_REFERENCE_LENGTH = 27;
    private static final int IPI_REFERENCE_LENGTH = 20;
    private static final Pattern NUMERIC = Pattern.compile("\\d+");
    private final Iso20022ChFieldFormatter formatter;

    public RemittanceResolver(Iso20022ChFieldFormatter formatter) {
        this.formatter = formatter;
    }

    public RemittanceInformation resolveRemittance(TransactionType transactionType, FinancePayment payment, IBankAccount bankAccount, TransactionType.LocalInstrument localInstrument, FinancePaymentSession session) throws BusinessDomainException {
        return transactionType.isDebitTransfer() ? this.resolveDebitRemittance(transactionType, payment, bankAccount, localInstrument, session) : this.resolveCreditRemittance(transactionType, payment);
    }

    private RemittanceInformation resolveDebitRemittance(TransactionType transactionType, FinancePayment payment, IBankAccount bankAccount, TransactionType.LocalInstrument localInstrument, FinancePaymentSession session) throws BusinessDomainException {
        Optional<RemittanceInformation> remittanceOpt;
        if (transactionType == TransactionType.TYPE_DD) {
            return this.createDefaultUnstructured(payment);
        }
        boolean lsvRaiffeisen = this.isLsvPlusRaiffeisenPerOpenItem(bankAccount, localInstrument, session);
        if (lsvRaiffeisen && (remittanceOpt = this.resolveLsvRaiffeisenRemittance(payment, bankAccount)).isPresent()) {
            return remittanceOpt.get();
        }
        return this.resolveDefaultDebitRemittance(payment, bankAccount);
    }

    private Optional<RemittanceInformation> resolveLsvRaiffeisenRemittance(FinancePayment payment, IBankAccount bankAccount) {
        Optional<String> paymentDetailsOpt = this.resolvePaymentDetails(payment);
        if (paymentDetailsOpt.isPresent()) {
            Optional<String> noteOpt = this.resolveNote(payment);
            return Optional.of(this.resolveRemittanceFromPaymentDetails(bankAccount, paymentDetailsOpt.get(), noteOpt));
        }
        Optional<String> noteOpt = this.resolveNote(payment);
        if (noteOpt.isPresent()) {
            return Optional.of(this.createUnstructured(noteOpt.get()));
        }
        Optional<String> referenceNumberOpt = this.resolveReferenceNumber(payment);
        if (referenceNumberOpt.isPresent()) {
            return Optional.of(this.createUnstructured(referenceNumberOpt.get()));
        }
        return Optional.empty();
    }

    private RemittanceInformation resolveDefaultDebitRemittance(FinancePayment payment, IBankAccount bankAccount) throws BusinessDomainException {
        Optional<String> paymentDetailsOpt = this.resolvePaymentDetails(payment);
        if (paymentDetailsOpt.isPresent()) {
            return this.resolveRemittanceFromPaymentDetails(bankAccount, paymentDetailsOpt.get(), Optional.empty());
        }
        return this.createDefaultUnstructured(payment);
    }

    private Optional<String> resolveNote(FinancePayment payment) {
        GeneratePaymentRemittance[] remittances = Iso20022ChPaymentUtils.getRemittances(payment);
        if (remittances != null && remittances.length == 1) {
            return this.resolveFromFirstRemittance(payment, RemittanceField.NOTE);
        }
        return Optional.empty();
    }

    private boolean isLsvPlusRaiffeisenPerOpenItem(IBankAccount bankAccount, TransactionType.LocalInstrument localInstrument, FinancePaymentSession session) {
        return localInstrument == TransactionType.LocalInstrument.LSV_PLUS && Iso20022ChPaymentUtils.isRaiffeisen(bankAccount) && Iso20022ChPaymentUtils.isPaymentPerOpenItemEnabled(session);
    }

    private Optional<String> resolvePaymentDetails(FinancePayment payment) {
        return this.resolveFromFirstRemittance(payment, RemittanceField.PAYMENT_DETAILS);
    }

    private Optional<String> resolveReferenceNumber(FinancePayment payment) {
        return this.resolveFromFirstRemittance(payment, RemittanceField.REFERENCE_NUMBER);
    }

    private Optional<String> resolveFromFirstRemittance(FinancePayment payment, RemittanceField field) {
        GeneratePaymentRemittance[] remittances = Iso20022ChPaymentUtils.getRemittances(payment);
        if (remittances == null || remittances.length == 0) {
            return Optional.empty();
        }
        String value = this.resolveValueFromRemittance(remittances[0], field);
        if (!StringMapper.isNullString((String)value)) {
            return Optional.of(value.trim());
        }
        return Optional.empty();
    }

    private String resolveValueFromRemittance(GeneratePaymentRemittance remittance, RemittanceField field) {
        OpenItem openItem = remittance.getOpenItem();
        switch (field) {
            case PAYMENT_DETAILS: {
                return openItem.getDirectDebitOrPaymentDetails();
            }
            case REFERENCE_NUMBER: {
                return openItem.getReferenceNumber();
            }
            case NOTE: {
                return openItem.getNote();
            }
        }
        return null;
    }

    private boolean isEsrMemberNumberPresent(IBankAccount bankAccount) {
        return !StringMapper.isNullString((String)bankAccount.getReferenceApplicationBACS());
    }

    private RemittanceInformation resolveRemittanceFromPaymentDetails(IBankAccount bankAccount, String paymentDetails, Optional<String> noteOpt) {
        if (this.isEsrMemberNumberPresent(bankAccount)) {
            return this.resolveEsrRemittance(paymentDetails, noteOpt);
        }
        return this.resolveIpiRemittance(paymentDetails, noteOpt);
    }

    private RemittanceInformation resolveEsrRemittance(String paymentDetails, Optional<String> noteOpt) {
        Optional<String> esrOpt = this.resolveEsrReference(paymentDetails);
        if (esrOpt.isPresent()) {
            return this.createStructuredDebit(TransactionType.ReferenceType.ESR, esrOpt.get(), noteOpt);
        }
        return this.createUnstructured(paymentDetails);
    }

    private Optional<String> resolveEsrReference(String value) {
        if (!NUMERIC.matcher(value).matches()) {
            return Optional.empty();
        }
        if (value.length() > 27) {
            return Optional.empty();
        }
        return Optional.of(StringUtils.leftPad((String)value, (int)27, (char)'0'));
    }

    private RemittanceInformation resolveIpiRemittance(String paymentDetails, Optional<String> noteOpt) {
        if (paymentDetails.length() == 20) {
            return this.createStructuredDebit(TransactionType.ReferenceType.IPI, paymentDetails, noteOpt);
        }
        return this.createUnstructured(paymentDetails);
    }

    private RemittanceInformation createStructuredDebit(TransactionType.ReferenceType referenceType, String reference, Optional<String> noteOpt) {
        String formattedReference = this.formatter.formatStructuredReference(reference);
        Optional<String> formattedNote = this.formatNote(noteOpt);
        return new RemittanceInformation(referenceType, formattedReference, formattedNote);
    }

    private Optional<String> formatNote(Optional<String> noteOpt) {
        if (noteOpt.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(this.formatter.formatUnstructuredRemittance(noteOpt.get()));
    }

    private RemittanceInformation resolveCreditRemittance(TransactionType transactionType, FinancePayment payment) throws BusinessDomainException {
        if (transactionType == TransactionType.TYPE_1) {
            return this.resolveEsrCredit(payment);
        }
        if (transactionType.equals((Object)TransactionType.TYPE_3)) {
            return this.resolveSwissCredit(payment);
        }
        return this.createDefaultUnstructured(payment);
    }

    private RemittanceInformation resolveEsrCredit(FinancePayment payment) {
        String esrNumber = payment.getEsrPayment().getEsrNumber();
        return this.createStructuredCredit(TransactionType.ReferenceType.ESR, esrNumber);
    }

    private RemittanceInformation resolveSwissCredit(FinancePayment payment) throws BusinessDomainException {
        SwissReferenceType swissReferenceType = payment.getSwissReferenceType();
        if (!SwissReferenceType.isStructuredReference((SwissReferenceType)swissReferenceType)) {
            return this.createDefaultUnstructured(payment);
        }
        TransactionType.ReferenceType referenceType = this.toReferenceType(swissReferenceType);
        String swissReference = payment.getSwissReference();
        return this.createStructuredCredit(referenceType, swissReference);
    }

    private TransactionType.ReferenceType toReferenceType(SwissReferenceType swissReferenceType) {
        switch (swissReferenceType) {
            case QR_REFERENCE: {
                return TransactionType.ReferenceType.QRR;
            }
            case ISO_CREDITOR_REFERENCE: {
                return TransactionType.ReferenceType.SCOR;
            }
        }
        return TransactionType.ReferenceType.UNSTRUCTURED;
    }

    private String defaultUnstructured(FinancePayment payment) throws BusinessDomainException {
        String remInfo = payment.isAdvancePayment() ? payment.getPaymentNote() : PaymentHelper.getFixedOrFreePaymentDetailsAsString((Payment)payment, 140, true, PaymentHelper.getPaymentDetailsHolder((PaymentSession)payment.getFinancePaymentSession()));
        if (StringMapper.isNullString((String)remInfo)) {
            remInfo = ".";
        }
        return remInfo;
    }

    private RemittanceInformation createUnstructured(String remittance) {
        String formatted = this.formatter.formatUnstructuredRemittance(remittance);
        return new RemittanceInformation(TransactionType.ReferenceType.UNSTRUCTURED, formatted, Optional.empty());
    }

    private RemittanceInformation createDefaultUnstructured(FinancePayment payment) throws BusinessDomainException {
        return this.createUnstructured(this.defaultUnstructured(payment));
    }

    private RemittanceInformation createStructuredCredit(TransactionType.ReferenceType referenceType, String reference) {
        String formatted = this.formatter.formatEsrNumber(reference);
        return new RemittanceInformation(referenceType, formatted, Optional.empty());
    }

    private static enum RemittanceField {
        PAYMENT_DETAILS,
        REFERENCE_NUMBER,
        NOTE;

    }
}

